<?php
// Path ke config disesuaikan
require_once '../config.php';

// GATEKEEPER: Proteksi halaman, hanya admin yang bisa akses
if (!isset($_SESSION['user_id']) || !isset($_SESSION['is_admin']) || $_SESSION['is_admin'] !== true) {
    header("Location: /admin/login");
    exit();
}

// =============================================================
// BAGIAN LOGIKA PROSES (TAMBAH, EDIT, HAPUS)
// =============================================================

// --- PROSES FORM (METHOD POST UNTUK TAMBAH & EDIT) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Ambil dan bersihkan input
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    $nama = trim($_POST['nama']);
    $url = trim($_POST['url']);
    $merchant = trim($_POST['merchant']);
    $apikey = trim($_POST['apikey']);
    $secret = trim($_POST['secret']);

    // Validasi sederhana
    if (empty($nama) || empty($url)) {
        $_SESSION['error_message'] = "Semua field wajib diisi.";
    } else {
        if ($id > 0) { // Jika ada ID, berarti ini adalah proses UPDATE (Edit)
            $stmt = $conn->prepare("UPDATE payment_setting SET nama = ?, url = ?, merchant = ?, apikey = ?, secret = ? WHERE id = ?");
            $stmt->bind_param("sssssi", $nama, $url, $merchant, $apikey, $secret, $id);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Payment gateway berhasil diperbarui.";
            } else {
                $_SESSION['error_message'] = "Gagal memperbarui data: " . $conn->error;
            }
        } else { // Jika tidak ada ID, berarti ini adalah proses INSERT (Tambah)
            $stmt = $conn->prepare("INSERT INTO payment_setting (nama, url, merchant, apikey, secret) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param("sssss", $nama, $url, $merchant, $apikey, $secret);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Payment gateway baru berhasil ditambahkan.";
            } else {
                $_SESSION['error_message'] = "Gagal menyimpan data: " . $conn->error;
            }
        }
        $stmt->close();
    }
    header("Location: payment-setting"); // Redirect kembali ke halaman utama untuk refresh
    exit();
}

// --- PROSES HAPUS (METHOD GET) ---
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $id_to_delete = intval($_GET['id']);
    $stmt = $conn->prepare("DELETE FROM payment_setting WHERE id = ?");
    $stmt->bind_param("i", $id_to_delete);
    if ($stmt->execute()) {
        $_SESSION['success_message'] = "Payment gateway berhasil dihapus.";
    } else {
        $_SESSION['error_message'] = "Gagal menghapus data: " . $conn->error;
    }
    $stmt->close();
    header("Location: payment-setting"); // Redirect kembali ke halaman utama untuk refresh
    exit();
}

// Menentukan tampilan berdasarkan parameter 'action' di URL
$action = $_GET['action'] ?? 'list'; // Default adalah 'list'

$page_title = "Pengaturan Payment Gateway";
require_once 'menu/header.php'; // Panggil header
?>

<div class="container">
    <main>
        <?php
        // =============================================================
        // TAMPILAN FORM (UNTUK TAMBAH ATAU EDIT)
        // =============================================================
        if ($action == 'tambah' || $action == 'edit'):
            $gateway_data = [ 'id' => '', 'nama' => '', 'url' => '', 'merchant' => '', 'apikey' => '', 'secret' => '' ];
            $form_title = "Tambah Payment Gateway Baru";

            // Jika action adalah 'edit', ambil data yang ada dari DB untuk diisi ke form
            if ($action == 'edit' && isset($_GET['id'])) {
                $id_to_edit = intval($_GET['id']);
                $stmt = $conn->prepare("SELECT * FROM payment_setting WHERE id = ?");
                $stmt->bind_param("i", $id_to_edit);
                $stmt->execute();
                $result = $stmt->get_result();
                if ($result->num_rows > 0) {
                    $gateway_data = $result->fetch_assoc();
                    $form_title = "Edit Payment Gateway";
                }
                $stmt->close();
            }
        ?>

        <div class="page-header">
            <h1><?php echo $form_title; ?></h1>
            <a href="payment-setting" class="btn btn-secondary">⬅️ Kembali ke Daftar</a>
        </div>

        <div class="card" style="margin-top: 2rem;">
            <form action="payment-setting" method="post">
                <?php if ($action == 'edit'): ?>
                    <input type="hidden" name="id" value="<?php echo htmlspecialchars($gateway_data['id']); ?>">
                <?php endif; ?>

                <div class="form-group">
                    <label for="nama">Nama Gateway</label>
                    <input type="text" id="nama" name="nama" class="form-control" value="<?php echo htmlspecialchars($gateway_data['nama']); ?>" required>
                </div>
                <div class="form-group">
                    <label for="url">URL Endpoint</label>
                    <input type="text" id="url" name="url" class="form-control" value="<?php echo htmlspecialchars($gateway_data['url']); ?>" required>
                </div>
                <div class="form-group">
                    <label for="merchant">Merchant ID / Code</label>
                    <input type="text" id="merchant" name="merchant" class="form-control" value="<?php echo htmlspecialchars($gateway_data['merchant']); ?>" >
                </div>
                <div class="form-group">
                    <label for="apikey">API Key / Client Key</label>
                    <input type="text" id="apikey" name="apikey" class="form-control" value="<?php echo htmlspecialchars($gateway_data['apikey']); ?>" >
                </div>
                <div class="form-group">
                    <label for="secret">Secret Key / Server Key</label>
                    <input type="password" id="secret" name="secret" class="form-control" value="<?php echo htmlspecialchars($gateway_data['secret']); ?>" >
                </div>
                <button type="submit" class="btn btn-primary">Simpan Perubahan</button>
            </form>
        </div>

        <?php
        // =============================================================
        // TAMPILAN DAFTAR (DEFAULT)
        // =============================================================
        else:
            // Ambil semua data payment gateway untuk ditampilkan di tabel
            $gateways = [];
            $result = $conn->query("SELECT * FROM payment_setting ORDER BY id DESC");
            if ($result) {
                while ($row = $result->fetch_assoc()) { $gateways[] = $row; }
            }
        ?>
        <div class="page-header">
            <h1>Daftar Payment Gateway</h1>
            <a href="?action=tambah" class="btn btn-primary">➕ Tambah Gateway Baru</a>
        </div>

        <?php
        // Tampilkan pesan notifikasi dari session jika ada
        if (isset($_SESSION['success_message'])) {
            echo '<div class="alert alert-success">' . htmlspecialchars($_SESSION['success_message']) . '</div>';
            unset($_SESSION['success_message']);
        }
        if (isset($_SESSION['error_message'])) {
            echo '<div class="alert alert-danger">' . htmlspecialchars($_SESSION['error_message']) . '</div>';
            unset($_SESSION['error_message']);
        }
        ?>

        <div class="card" style="margin-top: 2rem;">
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nama</th>
                            <th>Url</th>
                            <th>Merchant</th>
                            <th>API Key / Client Key</th>
                            <th>Secret Key / Server Key</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($gateways)): ?>
                            <tr>
                                <td colspan="5" style="text-align: center;">Belum ada data payment gateway.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($gateways as $gw): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($gw['id']); ?></td>
                                    <td><strong><?php echo htmlspecialchars($gw['nama']); ?></strong></td>
                                    <td><strong><?php echo htmlspecialchars($gw['url']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($gw['merchant']); ?></td>
                                    <td><?php echo substr(htmlspecialchars($gw['apikey']), 0, 15) . '...'; ?></td>
                                    <td><?php echo htmlspecialchars($gw['secret']); ?></td>
                                    <td>
                                        <a href="?action=edit&id=<?php echo $gw['id']; ?>" class="btn btn-sm btn-warning">Edit</a>
                                        <a href="?action=delete&id=<?php echo $gw['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Anda yakin ingin menghapus gateway ini?');">Hapus</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <?php endif; // Penutup dari if ($action == 'tambah' || $action == 'edit') ?>
    </main>
</div>

<?php
$conn->close(); // Tutup koneksi di akhir script
require_once 'menu/footer.php'; // Panggil footer
?>
