<?php
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
require_once '../config.php';

if (!isset($_SESSION['username'])) {
    header("Location: /login");
    exit();
}
$username = $_SESSION['username'];
$error_message = null;

// --- Definisikan Fee di Sini (INI BAGIAN YANG HILANG) ---
$biaya_tetap = 200;      // Contoh: fee tetap Rp 200
$persen_fee = 0.017;    // Contoh: fee 0.7% (ditulis 0.007)

// 2. CEK APAKAH ADA DEPOSIT YANG MASIH PENDING
// ==============================================
$stmt_check = $conn->prepare("SELECT id_depo FROM deposits WHERE username = ? AND status = 'Pending'");
$stmt_check->bind_param("s", $username);
$stmt_check->execute();
$result_check = $stmt_check->get_result();
if ($pending_deposit = $result_check->fetch_assoc()) {
    header("Location: payment?reff_id=" . $pending_deposit['id_depo']);
    exit();
}
$stmt_check->close();

// 3. PROSES FORM SAAT DI-SUBMIT
// ==================================
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $nominal_deposit = filter_input(INPUT_POST, 'nominal', FILTER_VALIDATE_INT);
    if (!$nominal_deposit || $nominal_deposit < 2000) {
        $error_message = "Jumlah deposit tidak valid. Minimum deposit adalah Rp 2000.";
    } else {
        
        // --- Kalkulasi Fee dan Total Bayar ---
        $fee_persen_amount = ceil($nominal_deposit * $persen_fee);
        $total_fee = $biaya_tetap + $fee_persen_amount;
        $total_bayar = $nominal_deposit + $total_fee;
        
        $gateway_name = 'atlantich2h';
        $stmt_gw = $conn->prepare("SELECT url, apikey FROM payment_setting WHERE nama = ?");
        $stmt_gw->bind_param("s", $gateway_name);
        $stmt_gw->execute();
        $gateway = $stmt_gw->get_result()->fetch_assoc();
        $stmt_gw->close();

        if (!$gateway) {
            $error_message = "Pengaturan payment gateway 'atlantich2h' tidak ditemukan.";
        } else {
            $reff_id = 'GP-' . strtoupper(uniqid());
            $payment_method = 'QRISFAST';
            
            $stmt_insert = $conn->prepare("INSERT INTO deposits (id_depo, username, nominal, fee, payment_method, status) VALUES (?, ?, ?, ?, ?, 'Pending')");
            $stmt_insert->bind_param("ssiis", $reff_id, $username, $nominal_deposit, $total_fee, $payment_method);
            
            if ($stmt_insert->execute()) {
                $stmt_insert->close();
                $api_url = rtrim($gateway['url'], '/') . '/deposit/create';
                $post_data = http_build_query(['api_key' => $gateway['apikey'], 'reff_id' => $reff_id, 'nominal' => $total_bayar, 'type' => 'ewallet', 'metode' => $payment_method]);
                
                $ch = curl_init();
                curl_setopt_array($ch, [CURLOPT_URL => $api_url, CURLOPT_POST => true, CURLOPT_POSTFIELDS => $post_data, CURLOPT_RETURNTRANSFER => true]);
                $response_body = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                $response_data = json_decode($response_body, true);
                
                if ($http_code == 200 && isset($response_data['status']) && $response_data['status'] == true && isset($response_data['data']['qr_image'])) {
                    $payment_image_url = $response_data['data']['qr_image'];
                    $expired_at = $response_data['data']['expired_at'];
                    $gateway_id = $response_data['data']['id'];
                    $stmt_update = $conn->prepare("UPDATE deposits SET payment_url = ?, expired_at = ?, gateway_id = ? WHERE id_depo = ?");
                    $stmt_update->bind_param("ssss", $payment_image_url, $expired_at, $gateway_id, $reff_id);
                    $stmt_update->execute();
                    $stmt_update->close();
                    header("Location: payment?reff_id=" . $reff_id);
                    exit();
                } else {
                    $stmt_fail = $conn->prepare("UPDATE deposits SET status = 'Failed' WHERE id_depo = ?");
                    $stmt_fail->bind_param("s", $reff_id);
                    $stmt_fail->execute();
                    $stmt_fail->close();
                    $error_message = "Gagal membuat permintaan deposit. Respon Gateway: <pre>" . htmlspecialchars($response_body) . "</pre>";
                }
            } else {
                $error_message = "Gagal menyimpan data deposit ke database.";
            }
        }
    }
}

// 4. Ambil Riwayat Deposit
// ==============================================
$history = [];
$stmt_history = $conn->prepare("SELECT id_depo, nominal, status, created_at, fee FROM deposits WHERE username = ? ORDER BY id DESC LIMIT 10");
$stmt_history->bind_param("s", $username);
$stmt_history->execute();
$result_history = $stmt_history->get_result();
while ($row = $result_history->fetch_assoc()) {
    $history[] = $row;
}
$stmt_history->close();

function format_tanggal_histori($date) {
    return date("d M Y, H:i", strtotime($date));
}

$page_title = "Deposit Saldo";
require_once 'header.php'; 
?>

<!--div class="container">
    <main>
        <div class="card-depo">
            <?php if ($error_message): ?>
                <div class="alert alert-danger"><?php echo $error_message; ?></div>
            <?php endif; ?>

            <form action="deposit" method="post">
                <div class="form-group">
                    <label for="nominal">Pilih Nominal</label>
                    <div class="nominal-pilihan">
                        <button type="button" class="btn-nominal" data-nominal="10000">Rp 10.000</button>
                        <button type="button" class="btn-nominal" data-nominal="25000">Rp 25.000</button>
                        <button type="button" class="btn-nominal" data-nominal="50000">Rp 50.000</button>
                        <button type="button" class="btn-nominal" data-nominal="100000">Rp 100.000</button>
                    </div>
                </div>
                <div class="form-group">
                    <label for="nominal-input">Atau Masukkan Jumlah Lain</label>
                    <input type="number" id="nominal-input" name="nominal" class="form-control" placeholder="Contoh: 15000" required>
                </div>
                <div class="nominal-display">
                    Anda akan membayar: <br><strong id="display-nominal">Rp 0</strong>
                </div>
                <button type="submit" class="btn btn-primary btn-block text-white">Lanjutkan ke Pembayaran</button>
            </form>
        </div>

        <div class="history-section">
            <h4>Riwayat Deposit</h4>
            <div class="card">
                <div class="table-responsive">
                    <table class="history-table">
                        <thead>
                            <tr>
                                <th>ID Deposit</th>
                                <th>Nominal</th>
                                <th>Status</th>
                                <th>Tanggal</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($history)): ?>
                                <tr>
                                    <td style="font-size:12px" colspan="4" style="text-align: center; padding: 2rem;">Belum ada riwayat deposit.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($history as $item): ?>
                                    <tr>
                                        <td style="font-size:12px" data-label="ID Deposit"><?php echo htmlspecialchars($item['id_depo']); ?></td>
                                        <td style="font-size:12px" data-label="Nominal"><small style="color: #8A94A6;"><?php echo htmlspecialchars($item['nominal']); ?>(Fee: <?php echo htmlspecialchars($item['fee']); ?>)</small>
                                        </td>
                                        <td style="font-size:12px" data-label="Status">
                                            <span class="status-badge status-<?php echo strtolower(htmlspecialchars($item['status'])); ?>">
                                                <?php echo htmlspecialchars($item['status']); ?>
                                            </span>
                                        </td>
                                        <td style="font-size:12px" data-label="Tanggal"><?php echo format_tanggal_histori($item['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

    </main>
</div-->
<div class="container my-4">
    <main>
        <div class="card shadow-lg rounded-4 p-4 mb-4">
            <h3 class="mb-3 text-center text-primary fw-bold" style="font-size:15px">💰 Deposit Saldo</h3>
            
            <?php if ($error_message): ?>
                <div class="alert alert-danger"><?php echo $error_message; ?></div>
            <?php endif; ?>

            <form action="deposit" method="post">
                <div class="mb-3">
                    <label style="font-size:12px"class="form-label fw-semibold">Pilih Nominal Cepat</label>
                    <div class="d-flex flex-wrap gap-2">
                        <button type="button" class="btn btn-outline-primary btn-nominal" data-nominal="10000">Rp 10.000</button>
                        <button type="button" class="btn btn-outline-primary btn-nominal" data-nominal="25000">Rp 25.000</button>
                        <button type="button" class="btn btn-outline-primary btn-nominal" data-nominal="50000">Rp 50.000</button>
                        <button type="button" class="btn btn-outline-primary btn-nominal" data-nominal="100000">Rp 100.000</button>
                    </div>
                </div>

                <div class="mb-3">
                    <label style="font-size:12px" for="nominal-input" class="form-label fw-semibold">Atau Masukkan Jumlah Lain</label>
                    <input style="font-size:12px" type="number" id="nominal-input" name="nominal" class="form-control form-control-lg rounded-3" placeholder="Contoh: 15000" required>
                </div>

                <div class="alert alert-info text-center fw-bold"style="font-size:12px">
                    Anda akan membayar: <br>
                    <span style="font-size:12px" id="display-nominal" class="text-success fs-4">Rp 0</span>
                </div>

                <button type="submit" class="btn btn-success w-100 py-2 rounded-3 shadow-sm">
                    Lanjutkan ke Pembayaran 🚀
                </button>
            </form>
        </div>

        <div class="history-section">
            <h4 style="font-size:15px" class="fw-bold text-secondary mb-3">📜 Riwayat Deposit</h4>
            <div class="card shadow-sm rounded-4">
                <div class="table-responsive">
                    <table class="table table-hover align-middle mb-0">
                        <thead style="font-size:12px" class="table-light">
                            <tr>
                                <th>ID Deposit</th>
                                <th>Nominal</th>
                                <th>Status</th>
                                <th>Tanggal</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($history)): ?>
                                <tr style="font-size:12px">
                                    <td style="font-size:12px" colspan="4" class="text-center py-4 text-muted">Belum ada riwayat deposit.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($history as $item): ?>
                                    <tr>
                                        <td style="font-size:12px"><?php echo htmlspecialchars($item['id_depo']); ?></td>
                                        <td style="font-size:12px">
                                            Rp <?php echo number_format($item['nominal'], 0, ',', '.'); ?>
                                            <br><small class="text-muted">Fee: Rp <?php echo number_format($item['fee'], 0, ',', '.'); ?></small>
                                        </td>
                                        <td style="font-size:12px">
                                            <?php 
                                                $status = strtolower($item['status']);
                                                $badgeClass = "secondary";
                                                if ($status == "pending") $badgeClass = "warning";
                                                if ($status == "success") $badgeClass = "success";
                                                if ($status == "failed") $badgeClass = "danger";
                                            ?>
                                            <span class="badge bg-<?php echo $badgeClass; ?> px-3 py-2">
                                                <?php echo htmlspecialchars($item['status']); ?>
                                            </span>
                                        </td>
                                        <td style="font-size:12px"><?php echo format_tanggal_histori($item['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

    </main>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const nominalButtons = document.querySelectorAll('.btn-nominal');
    const nominalInput = document.getElementById('nominal-input');
    const displayNominal = document.getElementById('display-nominal');
    
    const fixedFee = <?php echo $biaya_tetap; ?>;
    const percentFee = <?php echo $persen_fee; ?>;

    const updateDisplay = (amount) => {
        const number = parseInt(amount, 10) || 0;
        let totalAmount = 0;
        if (number > 0) {
            const percentageFeeAmount = Math.ceil(number * percentFee);
            totalAmount = number + fixedFee + percentageFeeAmount;
        }
        const formatted = new Intl.NumberFormat('id-ID', {
            style: 'currency',
            currency: 'IDR',
            minimumFractionDigits: 0
        }).format(totalAmount);
        displayNominal.textContent = formatted;
    };

    nominalButtons.forEach(button => {
        button.addEventListener('click', function() {
            nominalButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            const nominalValue = this.getAttribute('data-nominal');
            nominalInput.value = nominalValue;
            updateDisplay(nominalValue);
        });
    });

    nominalInput.addEventListener('input', function() {
        nominalButtons.forEach(btn => btn.classList.remove('active'));
        updateDisplay(this.value);
    });
});
</script>

<?php
require_once 'footer.php'; 
$conn->close();
?>